package org.coode.owlapi.obo.parser;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.semanticweb.owlapi.model.IRI;
import sun.misc.Regexp;


/**
 * Author: Matthew Horridge<br>
 * The University Of Manchester<br>
 * Bio-Health Informatics Group<br>
 * Date: 10-Jan-2007<br><br>
 */
public enum OBOVocabulary {



    DATA_VERSION("data-version"),
    VERSION("version"),
    DATE("date"),
    SAVED_BY("saved-by"),
    AUTO_GENERATED_BY("auto-generated-by"),
    ONTOLOGY("ontology"),
    SUBSETDEF("subsetdef"),
    IMPORT("import"),
    SYNONYM_TYPE_DEF("synonymtypedef"),
    SYNONYM_TYPE("synonymtype"),
    ID_SPACE("id_space"),
    DEFAULT_RELATIONSHIP_ID_PREFIX("default-relationship-id-prefix"),
    ID_MAPPING("id-mapping"),
    REMARK("remark"),
    ID("id"),
    NAME("name"),
    FORMAT_VERSION("format-version"),
    TYPEDEF("Typedef"),
    ALT_ID("alt_id"),
    ALT_NAME("alt_name"),
    NAMESPACE("namespace"),
    DEFAULT_NAMESPACE("default-namespace"),
    DEF("def"),
    COMMENT("comment"),
    SUBSET("subset"),
    SYNONYM("synonym"),
    RELATED_SYNONYM("related_synonym"),
    EXACT_SYNONYM("exact_synonym"),
    BROAD_SYNONYM("broad_synonym"),
    NARROW_SYNONYM("narrow_synonym"),
    XREF("xref"),
    XREF_ANALOGUE("xref_analogue"),
    XREF_UNKNOWN("xref_unk"),
    IS_A("is_a"),
    PART_OF("part_of"),
    RELATIONSHIP("relationship"),
    IS_OBSOLETE("is_obsolete"),
    REPLACED_BY("replaced_by"),
    CONSIDER("consider"),
    USE_TERM("use_term"),
    DOMAIN("domain"),
    RANGE("range"),
    IS_CYCLIC("is_cyclic"),
    IS_TRANSITIVE("is_transitive"),
    IS_SYMMETRIC("is_symmetric"),
    IS_ASYMMETRIC("is_asymmetric"),
    IS_REFLEXIVE("is_reflexive"),
    INVERSE("inverse"),
    TRANSITIVE_OVER("transitive_over"),
    INTERSECTION_OF("intersection_of"),
    UNION_OF("union_of"),
    DISJOINT_FROM("disjoint_from"),
    TERM("Term"),
    BUILTIN("builtin"),
    IS_METADATA_TAG("is_metadata_tag"),
    CARDINALITY("cardinality"),
    MAX_CARDINALITY("maxCardinality"),
    MIN_CARDINALITY("minCardinality"),
    INSTANCE("Instance"),
    INSTANCE_OF("instance_of"),
    PROPERTY_VALUE("property_value"),
    IS_ANONYMOUS("is_anonymous");

/**
     * @deprecated Use {@link #OBO_IRI_BASE}
     */
    @Deprecated
    public static final String ONTOLOGY_URI_BASE = "http://purl.org/obo/owl";

    /**
     * @deprecated Use {@link #OBO_IRI_BASE}
     */
    @Deprecated
    public static final String ANNOTATION_URI_BASE = "http://www.geneontology.org/formats/oboInOwl";


    public static final String OBO_IRI_BASE = "http://purl.obolibrary.org/obo/";


    public static final String LEGACY_OBO_IRI_BASE = "http://purl.org/obo/owl/";//

    /**
     * The pattern for OBO IDs.
     * Specified at <a href="http://www.obofoundry.org/id-policy.shtml">http://www.obofoundry.org/id-policy.shtml</a>
     */
    public static final Pattern OBO_ID_PATTERN = Pattern.compile("(([^:]+):)?(.+)");

    private static final String bases = Pattern.quote(OBO_IRI_BASE) + "|" + Pattern.quote(ONTOLOGY_URI_BASE + "/") + "|" + Pattern.quote(LEGACY_OBO_IRI_BASE) + "|" + Pattern.quote(ANNOTATION_URI_BASE + "/");

    public static final Pattern OBO_IRI_PATTERN = Pattern.compile("(" + bases + ")" + "(([^\\_]*)\\_)?([A-Za-z0-9\\_\\-]*)");

    



    /**
     * Converts OBO Ids to IRIs.  The conversion is defined at
     * <a href="http://www.obofoundry.org/id-policy.shtml">http://www.obofoundry.org/id-policy.shtml</a>
     * @param oboId The Id to convert
     * @return The IRI of the converted Id
     */
    public static IRI ID2IRI(String oboId) {
        Matcher matcher = OBO_ID_PATTERN.matcher(oboId);
        if(matcher.matches()) {
            String idSpace = matcher.group(2);
            String localId = matcher.group(3);
            StringBuilder sb = new StringBuilder();
            sb.append(OBO_IRI_BASE);
            if (idSpace != null) {
                sb.append(idSpace);
                sb.append("_");
            }
            sb.append(localId);
            return IRI.create(sb.toString());
        }
        else {
            return IRI.create(oboId);
        }
    }

//    Format of Foundry-compliant URIs
//
//    FOUNDRY_OBO_URI ::= "http://purl.obolibrary.org/obo/" IDSPACE "_" LOCALID
//    Format of OBO legacy URIs
//
//    Those are found in documents that were natively authored using the OBO format and which were converted using the NCBOoboInOwl script before this policy was put in place.
//
//    LEGACY_OBO_URI ::= "http://purl.org/obo/owl/" IDSPACE "#" IDSPACE "_" LOCALID

    public static String IRI2ID(IRI oboIRI) {
        Matcher matcher = OBO_IRI_PATTERN.matcher(oboIRI.toString());
        if(matcher.matches()) {
            String idSpace = matcher.group(3);
            String localId = matcher.group(4);
            StringBuilder sb = new StringBuilder();
            if (idSpace != null) {
                sb.append(idSpace);
                sb.append(":");
            }
            sb.append(localId);
            return sb.toString();
        }
        else {
            throw new RuntimeException("Not an OBO IRI");
        }
    }

    public static boolean isOBOIRI(IRI oboIRI) {
        return OBO_ID_PATTERN.matcher(oboIRI.toString()).matches();
    }



    private static final List<OBOVocabulary> headerTags =
            Arrays.asList(FORMAT_VERSION, DATA_VERSION, DATE, SAVED_BY,
                          AUTO_GENERATED_BY, SUBSETDEF, IMPORT, SYNONYM_TYPE_DEF,
                          ID_SPACE, DEFAULT_RELATIONSHIP_ID_PREFIX, ID_MAPPING,
                          REMARK);


    private static final List<OBOVocabulary> termStanzaTags =
            Arrays.asList(ID, NAME, NAMESPACE, ALT_ID,
                          DEF, COMMENT, SUBSET, SYNONYM, XREF,
                          IS_A, INTERSECTION_OF, UNION_OF,
                          DISJOINT_FROM, RELATIONSHIP,
                          IS_OBSOLETE, REPLACED_BY, CONSIDER);


    private static final List<OBOVocabulary> typeDefStanzaTags =
            Arrays.asList(ID, NAME, NAMESPACE, ALT_ID,
                          DEF, COMMENT, SUBSET, SYNONYM, XREF,
                          DOMAIN, RANGE, IS_ASYMMETRIC,
                          IS_CYCLIC, IS_REFLEXIVE, IS_SYMMETRIC,
                          IS_TRANSITIVE, IS_A, INVERSE, TRANSITIVE_OVER,
                          RELATIONSHIP, IS_METADATA_TAG,
                          IS_OBSOLETE, REPLACED_BY, CONSIDER);


    private static final List<OBOVocabulary> instanceStanzaTags =
            Arrays.asList(ID, NAME, NAMESPACE, ALT_ID,
                          DEF, COMMENT, SYNONYM, XREF,
                          INSTANCE_OF, PROPERTY_VALUE,
                          IS_OBSOLETE, REPLACED_BY, CONSIDER);


    OBOVocabulary(String name) {
        this.name = name;
    }

    private String name;


    public String getName() {
        return name;
    }


    public IRI getIRI() {
        return ID2IRI(name);
    }


    @Override
	public String toString() {
        return name;
    }


    public static List<OBOVocabulary> getHeaderTags() {
        return new ArrayList<OBOVocabulary>(headerTags);
    }


    public static List<OBOVocabulary> getTermStanzaTags() {
        return new ArrayList<OBOVocabulary>(termStanzaTags);
    }


    public static List<OBOVocabulary> getTypeDefStanzaTags() {
        return new ArrayList<OBOVocabulary>(typeDefStanzaTags);
    }


    public static List<OBOVocabulary> getInstanceStanzaTags() {
        return new ArrayList<OBOVocabulary>(instanceStanzaTags);
    }
}
