package org.semanticweb.owlapi.util;

import java.util.HashMap;
import java.util.Map;

import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLAnnotationProperty;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLDataProperty;
import org.semanticweb.owlapi.model.OWLDatatype;
import org.semanticweb.owlapi.model.OWLEntity;
import org.semanticweb.owlapi.model.OWLEntityVisitor;
import org.semanticweb.owlapi.model.OWLNamedIndividual;
import org.semanticweb.owlapi.model.OWLObjectProperty;


/**
 * Author: Matthew Horridge<br>
 * The University Of Manchester<br>
 * Bio-Health Informatics Group<br>
 * Date: 25-Nov-2007<br><br>
 * <p/>
 * Provides an entity URI conversion strategy which converts
 * entity URIs to a common base and alpha-numeric fragment.
 * The fragment is of the form An, where n is an integer (starting
 * at 1), and A is a string which depends on the type of entity:
 * <ul>
 * <li>For classes: A = "C"</li>
 * <li>For object properties: A = "op" </li>
 * <li>For data properties: A = "dp"</li>
 * <li>For individuals: A = "i"</li>
 * </ul>
 */
public class OWLEntityTinyURIConversionStrategy implements OWLEntityURIConverterStrategy {

    public static final String DEFAULT_BASE = "http://tinyname.org#";

    private String base;

    private Map<OWLEntity, IRI> entityNameMap;

    private OWLEntityFragmentProvider fragmentProvider;


    /**
     * Constructs an entity URI converter strategy, where the base
     * of the generated URIs corresponds to the value specified
     * by the DEFAULT_BASE constant.
     */
    public OWLEntityTinyURIConversionStrategy() {
        this(DEFAULT_BASE);
    }


    /**
     * Constructs an entity URI converter strategy, where the specified
     * base is used for the base of the URIs generated by the generator.
     *
     * @param base The base to be used.
     */
    public OWLEntityTinyURIConversionStrategy(String base) {
        this.base = base;
        entityNameMap = new HashMap<OWLEntity, IRI>();
        fragmentProvider = new OWLEntityFragmentProvider();
    }


    public IRI getConvertedIRI(OWLEntity entity) {
        IRI iri = entityNameMap.get(entity);
        if (iri != null) {
            return iri;
        }
        if (entity instanceof OWLDatatype) {
            return entity.getIRI();
        }
        String name = fragmentProvider.getName(entity);
        iri = IRI.create(base + name);
        entityNameMap.put(entity, iri);
        return iri;
    }


    private static class OWLEntityFragmentProvider implements OWLEntityVisitor {

        private String name;

        private int classCount = 0;

        private int objectPropertyCount = 0;

        private int dataPropertyCount = 0;

        private int individualCount = 0;

        private int annotationPropertyCount = 0;

        private int datatypeCount = 0;


        public String getName(OWLEntity entity) {
            if (entity.isBuiltIn()) {
                return entity.getIRI().toString();
            }
            entity.accept(this);
            return name;
        }


        @SuppressWarnings("unused")
		public void visit(OWLClass cls) {
            classCount++;
            name = "C" + classCount;
        }

        @SuppressWarnings("unused")
        public void visit(OWLDatatype datatype) {
            datatypeCount++;
            name = "dt" + datatypeCount;
        }

        @SuppressWarnings("unused")
        public void visit(OWLNamedIndividual individual) {
            individualCount++;
            name = "i" + individualCount;
        }

        @SuppressWarnings("unused")
        public void visit(OWLDataProperty property) {
            dataPropertyCount++;
            name = "dp" + dataPropertyCount;
        }

        @SuppressWarnings("unused")
        public void visit(OWLObjectProperty property) {
            objectPropertyCount++;
            name = "op" + objectPropertyCount;
        }
        @SuppressWarnings("unused")
        public void visit(OWLAnnotationProperty property) {
            annotationPropertyCount++;
            name = "ap" + annotationPropertyCount;
        }
    }
}
